#ifndef _DT_WINDOWS_SUPPORT_
#define _DT_WINDOWS_SUPPORT_
// assume the following already included
//#include <windows.h>


//------------------------------------------------------------------------------------------
class HandleWrapper
{
public:
	HandleWrapper(HANDLE h_ = NULL) { h = h_; }
	~HandleWrapper() { if(h) CloseHandle(h); }

	HandleWrapper& operator = (HANDLE h_)
	{
		if(h) CloseHandle(h);
		h = h_;
		return *this;
	}

	void throwIfNull() { if(!h) throw 0; }

	// cast to "HANDLE"
	operator HANDLE () { return h; }
	// cast to bool
	operator bool () const { return h != 0; }
	// not operator
	bool operator !() const { return h == 0; }

protected:
	HANDLE h;

private: // multiple owners not supported
	void operator = (const HandleWrapper& h_) { throw 0; }
	HandleWrapper(const HandleWrapper& h_) { throw 0; }
};

//------------------------------------------------------------------------------------------
class MutexWrapper : public HandleWrapper
{
public:
	MutexWrapper(
		LPSECURITY_ATTRIBUTES security=NULL, BOOL initial_owner=FALSE, LPCTSTR name=NULL) :
			HandleWrapper(CreateMutex(security, initial_owner, name)) { throwIfNull(); }
};

//------------------------------------------------------------------------------------------
class ScopeMutexLock
{
public:
	HANDLE mutex;
	ScopeMutexLock(HANDLE mutex_, int timeout_msec=2000)
	{
		if(WaitForSingleObject(mutex_, timeout_msec) == WAIT_OBJECT_0) {
			mutex = mutex_;
		}
		else {
			mutex = 0;
			throw 0;
		}
	}
	~ScopeMutexLock() { if(mutex) ReleaseMutex(mutex); }
};

//------------------------------------------------------------------------------------------
class CriticalSectionWrapper : public CRITICAL_SECTION
{
public:
	CriticalSectionWrapper() { InitializeCriticalSection(this); }
	~CriticalSectionWrapper() { DeleteCriticalSection(this); }
	operator CRITICAL_SECTION* () { return this; }
};
//------------------------------------------------------------------------------------------
class ScopeCriticalSection
{
public:
	CRITICAL_SECTION* cs;
	ScopeCriticalSection(CRITICAL_SECTION* cs_)
	{
		cs = cs_;
		EnterCriticalSection(cs_);
	}
	~ScopeCriticalSection() { LeaveCriticalSection(cs); }
};

//-------------------------------------------------------------------------------------------------
class RefCountObj
//
// Always inherit reference counted objects from "RefCountObj" as virtual
//
{
public:
	mutable int _ref_count;
	RefCountObj(int ref_count=0) { _ref_count=ref_count; }

	static CRITICAL_SECTION* protection()
	{
		static CriticalSectionWrapper _t;
		return _t;
	}
};

//-------------------------------------------------------------------------------------------------
template <class X> class Ref
//
// reference counting template
//
{
public:
	Ref(X *x = 0) { _o = 0; set(x); }
	~Ref() { set(0); } // note, not virtual

	void set(X* x)
	{
		X* old = 0;
		{
			ScopeCriticalSection lock(RefCountObj::protection());

			// store the old pointer if not on manual delete
			if(_o && _o->RefCountObj::_ref_count != -1) old = _o;

			// incr new object first in case new & old are the same
			_o = x;
			if(_o && _o->RefCountObj::_ref_count != -1)
				_o->RefCountObj::_ref_count++;

			// decr old object's ref count
			if(old) {
				old->RefCountObj::_ref_count--;
				if(old->RefCountObj::_ref_count > 0) old = 0;
				// if old ref count is 0, pointer is still set and is deleted
				// out of scope of mutex
			}
		}
		// out of mutex scope, delete old if need be
		if(old) delete old;
	}

	// pointer operator
	X* operator ->() const { return _o; }
	// auto type-cast back to pointer
	operator X* () const { return _o; }
	// use () operator to explicitly return pointer
	X* operator () () const { return _o; }
	// deref operator
	X& operator * () const { return *_o; }
	// assignment operators
	Ref& operator = (X* src) { set(src); return *this; }
	Ref& operator = (const Ref& src) { set(src); return *this; }
	// compare
	bool operator == (const X* b) const { return _o == b; }
	// cast to bool
	operator bool () const { return _o != 0; }
	// not operator
	bool operator! () const { return _o == 0; }

protected:
	X* _o;
};

//-------------------------------------------------------------------------------------------------
template <class X> class RefNew : public Ref<X>
//
// allocate a new instance of "X" on creation
// i.e. instead of
//
//   Ref<MyClass> my_class = new MyClass;
//
// do this:
//   RefNew<MyClass> my_class;
{
public:
	RefNew() { set(new X); }
	RefNew& operator = (X* src) { set(src); return *this; }
	RefNew& operator = (const Ref<X>& src) { set(src); return *this; }
};

#endif
