/******************************************************************************
DtSliderBase


History
	Date       Version    Programmer         Comments
	29/07/03   1.0        Darrell Tam
******************************************************************************/

#if !defined(AFX_DtSliderBase_H__4819D6EB_ADEE_436D_A222_AF6736E2F14D__INCLUDED_)
#define AFX_DtSliderBase_H__4819D6EB_ADEE_436D_A222_AF6736E2F14D__INCLUDED_

#include <vector>
#include "DtCtrl.h"
#include "DtMultiEdit.h"

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000
// DtSliderBase.h : header file
//

/////////////////////////////////////////////////////////////////////////////
// macro to map all slider messages to functions
//
#define DT_SLIDER_MSGS(ID, FN_PREFIX) \
	ON_CONTROL(DtSliderBase::NC_REDRAW, ID, FN_PREFIX##Redraw) \
	ON_CONTROL(DtSliderBase::NC_USERCHG, ID, FN_PREFIX##UserChg) \
	ON_CONTROL(DtSliderBase::NC_EDITTXTCHG, ID, FN_PREFIX##EditTxtChg) \
	ON_CONTROL(DtSliderBase::NC_KEYPRESS, ID, FN_PREFIX##KeyPress)

/////////////////////////////////////////////////////////////////////////////
// DtSliderBase window

class DtSliderBase : public DtCtrl
{
public:
	DtSliderBase();
	virtual ~DtSliderBase();

protected:
	// derived class must call this
	void setNumSliders(int n, bool alloc_sliders=true);

// Attributes
public:
	// notify codes sent to parent
	enum {
		NC_EDITTXTCHG = DtCtrl::NC_LAST, // user has changed text edit
		NC_USERCHG,		// user has moved position of knob or changed text edit 
		NC_KEYPRESS,	// intercept key presses
		NC_LAST
	} NOTIFY_CODES;

public:
	// methods to access multi-edits (each slider has a corresponding multi-edit)
	DtMultiEdit& multiEdit(int slider) { return s[slider]->medit; }
	DtMultiEdit& operator () (int slider) { return multiEdit(slider); }

	// return multi-edit belonging to active slider
	DtMultiEdit& multiEdit() { int active=getActive(); if(active==-1) throw 0; return s[active]->medit; }
	DtMultiEdit& operator () (void) { return multiEdit(); }

	// set the edit text at <edit_num> if the user isn't typing into it
	bool setEditText(int slider, int edit_num, const string& s, bool force=false);
	bool operator () (int slider, int edit_num, const string& s, bool force=false) { return setEditText(slider,edit_num,s,force); }

	// return which slider is active
	int/*slider*/getActive(void);

	// get value from slider "i"
	float getVal(int slider) { return s[slider]->value; }

	// set value without redraw
	virtual void setVal(int slider, float v)
		{ s[slider]->value = limit_range(v, 0.0f, 1.0f); }

	// check whether user is changing slider
	bool isUserChanging(int slider)
		{ return rbutton_down || s[slider]->isActive()&&(lbutton_down || edit_lock); }
	
public: // calls made by control parent
	// called by parent with "v" array containing rounded values for display
	virtual bool/*redraw*/chkRedraw(float/*0..1*/* v);

	// enable key intercept
	void setParentInterceptKeys(bool x) { parent_intercept_keys = x; }

	// called by the parent during key intercept to consume the key press
	void cancelKeyPress() { cancel_key_press = true; }

protected: // helpers to be called from derived classes

	// call during drawControl() to update multi edit colour and font
	void prepareDrawText(CSize* /*out*/ sz);

	// draw multi-edit cursor for active multi-edit (call prepareDrawText() first)
	void drawCursor(CPoint* p);

	// draw all multi-edit text (call prepareDrawText() first)
	void drawText(CPoint* p);

	// select a particular slider, no auto redraw
	void selectSlider(int n);

protected:
	// determine which slider is to be selected based on mouse position
	virtual void selectSlider(void) = 0;

protected:	// DtCtrl overrides (call from parent if parentDraw)

	// derived class needs to draw the sliders here
	virtual void drawControl(void) = 0;

	virtual bool chkRedraw(void);

protected:	// DtCtrl overrides

	virtual void OnMouseLeave(void);
	virtual void OnLButtonDrag(void);
	virtual void OnRButtonDrag(void);
	virtual void OnMouseMove(void);
	virtual void OnLButtonDblClk(void);
	virtual void OnLButtonDown(void);
	virtual void OnLButtonUp(void);
	virtual void OnRButtonDown(void);
	virtual void OnRButtonDblClk(void);
	virtual void OnRButtonUp(void);

protected: // Internal stuff
	float mouseVal(void) { return (float)curr_point.x/(client_rect.Width()-1); }
	float getValPixel(int slider) { return getVal(slider)*(client_rect.Width()-1); }

	// access colour of slider based on state
	COLORREF knobStateColour(int slider);
	COLORREF editStateColour(int slider);

	// each slider has this structure
	struct Slider
	// structure to store details of each slider
	{
		float value;			// current value

		DtMultiEdit medit;		// text edits

		typedef enum { INACTIVE, TIMER_NUDGE, DIRECT_ADJUST, N_STATES } knob_state_;

		TChk<knob_state_> knob_state;
		TChk<long> screen_pos;	// position on screen

		float click_delta;		// temporary value used during drag

		bool draw_text_left;	// text position

		Slider();
		virtual ~Slider();

		void deselect(void);
		void select(CPoint p);
		void nudge(float mouse_val, float move_frac);
		void drag(float mouse_val);
		void setClickDelta(int x, int window_width);
		bool isActive(void) { return knob_state != INACTIVE; }
	};

	// each control has 1 or more sliders
	vector<auto_ptr<Slider> > s;

	// user is typing, don't change values of active edit
	bool edit_lock;

	// lookup tables for colour indexed by Stuff:knob_state
	COLORREF knob_state_colour[Slider::N_STATES];
	COLORREF edit_state_colour[Slider::N_STATES];

	// set to true if parent gets key presses first
	bool parent_intercept_keys;

	// set true by parent during key intercept
	bool cancel_key_press;

	// temporary space
	vector<float> _temp;

	// Generated message map functions
protected:
	//{{AFX_MSG(DtSliderBase)
	afx_msg void OnTimer(UINT nIDEvent);
	afx_msg void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

/////////////////////////////////////////////////////////////////////////////

//{{AFX_INSERT_LOCATION}}
// Microsoft Visual C++ will insert additional declarations immediately before the previous line.

#endif // !defined(AFX_DtSliderBase_H__4819D6EB_ADEE_436D_A222_AF6736E2F14D__INCLUDED_)
