/******************************************************************************

ParamDesc
	Code for individual param descriptions

History
	Date       Version    Programmer         Comments
	28/9/03    1.0        Darrell Tam		Created
******************************************************************************/


#include <StdAfx.h>
#include "stdlib.h"
#include "string.h"

#include "params.h"
#include "ParamDesc.h"
#include "MIBlkFx.h"
//**********************************************************************************************

//------------------------------------------------------------------------------------------
static class ParamDescHex : public ParamDesc
// Hex conversion
{
public:
	// pre & post units for description "i"
	const char* prefix() const { return "0x"; }

	// return text value for v
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
	{
		FxValTrkParam p;
		p.set(v);
		sprintf(txt_out, "%04x", p.get());
	}

	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		FxValTrkParam ret;
		int h; if(sscanf(txt_in, "%x", &h)!=1) return -1.0f;
		ret.set(h);
		return ret.getFrac();
	}

} _hex;
ParamDesc* param_desc_hex = &_hex;

//------------------------------------------------------------------------------------------
static class ParamDescPercent : public ParamDesc
// 0 .. 100%
{
public:
	// pre & post units for description "i"
	const char* postfix() const { return "%"; }

	// return text value for v
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
		{ sprintf(txt_out, "%.0f", v*100.0f);	}

	static bool parse(const char* txt_in, float* /*out*/percent)
		{ return sscanf(txt_in, "%f", percent)==1; }

	// given description text, convert to value (return -1 if can't convert)
	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		float percent;
		return parse(txt_in, &percent)?percent*0.01f:-1.0f;
	}

} _percent;
ParamDesc* param_desc_percent = &_percent;

//------------------------------------------------------------------------------------------
static class ParamDescPercent2 : public ParamDescPercent
// -100% .. 100%
{
public:
	// return text value for v
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
		{ ParamDescPercent::desc(txt_out, v*2.0-1.0f, mi); }

	// given description text, convert to value (return -1 if can't convert)
	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		float percent;
		return parse(txt_in, &percent)?(percent*0.01f+1.0f)/2.0f:-1.0f;
	}
} _percent2;
ParamDesc* param_desc_percent2 = &_percent2;

//------------------------------------------------------------------------------------------
static class ParamDescFreqShift : public ParamDesc
// frequency shift, -22050 to 22050 Hz using an arbitary linear+exp function
{
	
public:
	// pre & post units for description "i"
	const char* postfix() const { return "Hz"; }

	// return text value for v (Hz)
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
	{
		sprintf(txt_out, "%.0f", mi.roundFreqDisp(freqShift(v)));
	}

	// given description text in Hz, convert to value (return -1 if can't convert)
	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		float freq;
		if(sscanf(txt_in, "%f", &freq)!=1) return -1.0f;

		// since I chose a difficult to invert function, solve with a binary search
		float r_min = 0.0f, r_max = 1.0f;
		float m;
		while(r_max-r_min > 1e-6) {
			m = (r_max+r_min)/2.0f;
			float t = freqShift(m);
			if(t > freq) r_max = m; else r_min = m;
		}
		return m;
	}
public:
} _freq_shift;
ParamDesc* param_desc_freq_shift = &_freq_shift;

//------------------------------------------------------------------------------------------
static class ParamDescHarmonicWidth : public ParamDescPercent
//
// Harmonic width (percent)
//
{
public:
	
	// return text value for v
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
	{ ParamDescPercent::desc(txt_out, SplitHarmonicParam(v).width_frac, mi); }

	// given description text, convert to value (return -1 if can't convert)
	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		float prev = mi.getParamsDisp().tv[trk_num].fxval;
		SplitHarmonicParam p(prev);
		float percent;
		if(!ParamDescPercent::parse(txt_in, &percent)) return -1.0f;
		p.width_frac = limit_range(percent*.01f, 0.0f, 1.0f);
		return p;
	}
} _harmonic_width;
ParamDesc* param_desc_harmonic_width = &_harmonic_width;

//------------------------------------------------------------------------------------------
static class ParamDescHarmonicType : public ParamDesc
// 
// type is 0..2 (b, o, e)
//
{
public:
	// return text value for v
	void desc(char* txt_out, float/*0..1*/v, MIBlkFx& mi) const
	{
		static const char* harm_type_name[] = {"both", "odd", "even"};
		strcpy(txt_out, harm_type_name[SplitHarmonicParam(v).harm_type]);
	}

	// given description text, convert to value (return -1 if can't convert)
	float/*0..1, -1=error*/param(const char* txt_in, MIBlkFx& mi, int trk_num) const
	{
		float prev = mi.getParamsDisp().tv[trk_num].fxval;
		SplitHarmonicParam p(prev);
		switch(txt_in[0]) {
			case 'b': p.harm_type = 0; break; // both
			case 'o': p.harm_type = 1; break; // odd
			case 'e': p.harm_type = 2; break; // even
			default: return -1.0f;
		}
		return p;
	}
} _harmonic_type;
ParamDesc* param_desc_harmonic_type = &_harmonic_type;

/*
//**********************************************************************************************
namespace AmpParamDesc {
	ParamDesc* param_desc_hex = &ParamDescHex<_AmpParam>::inst();

	//------------------------------------------------------------------------------------------
	static class Decibels : public ParamDescT<_AmpParam>
	{
	public:
		const char* postfix() const { return "dB"; }

		void desc(char* txt_out, _AmpParam p, MIBlkFx& mi) const
		{ sprintf(txt_out, p.get()>0?"%.1f":"-inf", p.getdB()); }

		void param(const char* txt_in, _AmpParam& ret, MIBlkFx& mi, int trk_num) const
		{
			if(strncmp(txt_in, "-i", 2)==0) ret.set(0);
			else {
				float dB;
				if(sscanf(txt_in, "%f", &dB)!=1) return;
				ret.setdB(dB);
			}
		}
	} _decibels;
	ParamDesc* param_desc_decibels = &_decibels;

	//------------------------------------------------------------------------------------------
	static class ParamDescPercent : public ParamDescT<_AmpParam>
	{
	public:
		const char* postfix() const { return "%"; }

		void desc(char* txt_out, _AmpParam p, MIBlkFx& mi) const
			{ sprintf(txt_out, "%.1f", p.getMult()*100.0f); }

		void param(const char* txt_in, _AmpParam& ret, MIBlkFx& mi, int trk_num) const
		{
			float percent;
			if(sscanf(txt_in, "%f", &percent)!=1) return;
			ret.setMult(percent/100.0f);
		}
	} _percent;
	ParamDesc* param_desc_percent = &_percent;
}

//**********************************************************************************************
namespace FreqParamDesc {
	//------------------------------------------------------------------------------------------
	static class FreqHz : public ParamDescT<_FreqParam>
	{
	public:
		const char* postfix() const { return "Hz"; }

		void desc(char* txt_out, _FreqParam p, MIBlkFx& mi) const
			{ sprintf(txt_out, "%.1f", p.getHz(); }

		void param(const char* txt_in, _FreqParam& ret, MIBlkFx& mi, int trk_num) const
		{
			float percent;
			if(sscanf(txt_in, "%f", &percent)!=1) return;
			ret.setMult(percent/100.0f);
		}
	} _freq_hz;
	ParamDesc* FreqHz = &_freq_hz;
}
*/