// Spectrogram.cpp : implementation file
//

#include "stdafx.h"
#include <float.h>
#include "misc_stuff.h"
#include "Spectrogram.h"
#include "params.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//------------------------------------------------------------------------------------------
BEGIN_MESSAGE_MAP(Spectrogram, CWnd)
	//{{AFX_MSG_MAP(Spectrogram)
	ON_WM_PAINT()
	ON_WM_MOUSEMOVE()
	ON_WM_CREATE()
	ON_WM_TIMER()
	ON_WM_KILLFOCUS()
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_SCROLLDRAW, handleScrollDraw)	
	ON_MESSAGE(WM_SETFONT, handleSetFont)
END_MESSAGE_MAP()

//------------------------------------------------------------------------------------------
void Spectrogram::ColMap::generate(Array<float, 4>* col_vec)
{
	Array<float, 3> a, b;

	float frac, frac_step;
	int end_i = 0, beg_i;

	int n = SIZE;

	b.copy(&((*col_vec)[1]));

	for(int i = 0; i < SIZE; i++) {
		if(i >= end_i) {
			beg_i = end_i;
			a = b;
			col_vec++;
			end_i = _cpp_max((int)((*col_vec)[0]*SIZE), beg_i+1);
			b.copy(&((*col_vec)[1]));
			frac_step = 1.0f/(end_i-beg_i);
			frac = 0.0f;
		}
		Array<float, 3> result;
		for(int k = 0; k < 3; k++) result[k] = lin_interp(frac, a[k], b[k]);

		cRGB t(result); // have to reverse order...
		data[i] = cRGB(result).swapRB().col;

		frac += frac_step;
	}
}


// spectrogram value scaling
//------------------------------------------------------------------------------------------
Spectrogram::Spectrogram() : horiz_line(-1), vert_line(-1)
{
	timer_redraw_pending = false;

	control_active = false;
	scrolling_in_progress = false;
	pixel_freq_map = NULL;
	prev_disp_samp_pos = 0;
	min_disp_step = 4000;

	val_min = powf(1e-4f,2.0f);
	val_max = powf(32768*1e-3f,2.0f);
	col_log_offs = logf(val_min);
	col_log_scale = ColMap::SIZE/(logf(val_max)-col_log_offs);

}

//------------------------------------------------------------------------------------------
Spectrogram::~Spectrogram()
{
}


//------------------------------------------------------------------------------------------
inline unsigned long Spectrogram::colourMap(float in)
{
	int out;
	if(in > val_min) {
		out = (int)((logf(in)-col_log_offs)*col_log_scale+0.5f);
		if(out >= ColMap::SIZE) out = ColMap::SIZE-1;
	}
	else out = 0;
	return col_map[out];
}

		
//------------------------------------------------------------------------------------------
void Spectrogram::setPixelFreqMap(PixelFreqMap* x, int height)
{
	pixel_freq_map = x;
	int w = x->getNPixels()-1;

	image_i = 0;
	image_width = w;
	image_height = height;
	image_data.resize(image_width*image_height);
	memset(image_data.begin(), 0, byte_sizeof(image_data));
	curr_min.resize(w);
	curr_max.resize(w);

	image_line_in = 0;
	image_line_disp.setBoth(0);
	curr_line_is_empty = true;

	abs_samp_pos.resize(image_height);
	memset(abs_samp_pos.begin(), 0, byte_sizeof(abs_samp_pos));
}

//----------------------------------------------------------------------------------------------------
LRESULT Spectrogram::handleScrollDraw(WPARAM wParam/*not used*/, LPARAM lParam/*not used*/)
// windows message to draw
{
	// check for anything to do
	int scroll_lines = image_line_disp.next-image_line_disp;
	image_line_disp.update();
	if(!scroll_lines && !horiz_line.isChanged() && !vert_line.isChanged()) return Default();

	CClientDC dc(this);

	// remove lines if required
	if(horiz_line.isChanged() || scroll_lines) dc.InvertRect(&horizLineRect());
	if(vert_line.isChanged() || scroll_lines) dc.InvertRect(&vertLineRect());

	// scroll if need be
	if(scroll_lines) {
		// always scroll upwards
		if(scroll_lines < 0) scroll_lines += image_height;
		scrolling_in_progress = true;
		ScrollWindow(0, -scroll_lines);
		scrolling_in_progress = false;

	}

	// update lines
	if(horiz_line.update() || scroll_lines)	dc.InvertRect(&horizLineRect());
	if(vert_line.update() || scroll_lines) dc.InvertRect(&vertLineRect());

	return Default();
}

//------------------------------------------------------------------------------------------
void Spectrogram::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	CRect client_rect;
	GetClientRect(client_rect);

	// limit display height to buffer height
	if(client_rect.Height() > image_height) {
		// fill rectangle for which there is no data with black
		CRect r = client_rect;
		client_rect.top += client_rect.Height()-image_height;

		r.bottom = client_rect.top;
		if(r.Height() > 0) dc.FillRect(r, &CBrush(0UL));
	}

	// create a memory dc containing the image data
	CDC mdc;
	mdc.CreateCompatibleDC(&dc);
	CBitmap bm;
	bm.CreateBitmap(image_width, image_height, 1, 32, image_data.begin());
	ScopeSelect<CBitmap> ss(&mdc, &bm);

	// perform copy in at most 2 blt's (depending on whether we need to wrap in src buffer)
	//
	// rely on dc's clip region to optimize data blt'd
	//
	CRect r = client_rect;
	while(1) {
		int n_lines = r.Height();
		if(n_lines <= 0) break;

		int src_start_line = image_line_disp-client_rect.Height()+r.top;
		if(src_start_line < 0) {
			// wrap to end of source buffer
			src_start_line += image_height;

			// limit number of lines to end-of-buffer
			n_lines = min(image_height-src_start_line, n_lines);
		}

		// do the copy
		dc.BitBlt(r.left, r.top, r.Width(), n_lines, &mdc, r.left, src_start_line, SRCCOPY);
		r.top += n_lines;
	}

	// draw lines unless scrolling
	if(!scrolling_in_progress) {
		dc.InvertRect(&horizLineRect());
		dc.InvertRect(&vertLineRect());
	}
}

//------------------------------------------------------------------------------------------
void Spectrogram::setHorizLine(long horiz_line_pos_abs)
{
	int offs = image_line_disp; // index into abs sample position vector

	int beg = 0;
	int end = offs-image_line_in;
	if(end < 0) end += image_height;

	int i = 0; // unwrapped index (starting at "s")

	// binary search for where line should go
	long diff;
	TChk<int> mid(-1);
	while(1) {
		mid = (beg+end)/2;
		if(!mid.update()) break;

		// convert to direct index
		int k = offs-mid; 
		if(k < 0) k += image_height;

		// find which segment to use
		diff = horiz_line_pos_abs-abs_samp_pos[k];
		if(diff > 0) end = mid; else beg = mid;
	}

	// update display line position
	CRect r; GetClientRect(/*out*/r);
	horiz_line = r.bottom-mid - (diff<0?2:1);
	if(horiz_line.isChanged()) PostMessage(WM_SCROLLDRAW, 0, 0);
}


//------------------------------------------------------------------------------------------
inline float safe_logf(float x)
{
	if(x > 0.0f) return logf(x);
	return -88.0f; // logf(FLT_MIN) = -87.3365
}

//------------------------------------------------------------------------------------------
inline void Spectrogram::getMinMax(bool empty_line, int n, pcplxf in)
{
	int w = image_width;
	const PixelFreqMap::BinRng* p_to_bin = pixel_freq_map->pixelToBinRng(n);

	for(int i = 0; i < w; i++) {
		float min_val = empty_line?FLT_MAX:curr_min[i];
		float max_val = empty_line?0:curr_max[i];

		// find min/max power in each pixel range
		PixelFreqMap::BinRng br = p_to_bin[i];

		float* seg_end = in.r+br.end;
		pcplxf x(in.r+br.start, in.i-br.start);
		while(x.r < seg_end) {
			float v = norm(*x++);
			max_val = max(max_val, v);
			min_val = min(min_val, v);
		}
		curr_min[i] = min_val;
		curr_max[i] = max_val;
	}
}

//------------------------------------------------------------------------------------------
void Spectrogram::newData(long abs_pos, int/*log2(fft_blk_len/BLK_SZ_0)*/n, pcplxf in)
{

	// output image width
	int w = image_width;

	if(curr_line_is_empty) {
		curr_line_is_empty = false;
		abs_samp_pos[image_line_in] = abs_pos;
		getMinMax(/*empty_line*/true, n, in);
	}
	else {
		getMinMax(/*empty_line*/false, n, in);
	}

	// limit maximum scroll speed
	if(abs_pos - prev_disp_samp_pos < min_disp_step) return;

	// render spectrogram line
	unsigned long *out_ptr = &image_data[image_line_in*image_width];
	float *val_ptr = &curr_max[0];
	for(int i = 0; i < w; i++, out_ptr++, val_ptr++)
		*out_ptr = colourMap(*val_ptr);

	prev_disp_samp_pos = abs_pos;
	image_line_disp = image_line_in;

	// circular buffer
	if(image_line_in >= image_height-1) image_line_in = 0;
	else image_line_in++;
	curr_line_is_empty = true;

	// send draw message
	PostMessage(WM_SCROLLDRAW, 0, 0);
}

//------------------------------------------------------------------------------------------
void Spectrogram::updateLabel()
// update the label window (if control is active)
{
	if(!control_active) return;

	// update label (window above sgram)
	float hz = pixel_freq_map->pixelToFreq(vert_line);
	_FreqParam param; param.setHz(hz);

	char _buf[100]; char* buf = _buf;
	buf += sprintf(buf, "%s %5.0f Hz / ", name.c_str(), hz);
	buf += HzToNote(buf, hz);
	buf += sprintf(buf, " / 0x%04X", (long)param);

	CRect r; GetWindowRect(r);
	r.bottom = r.top+1; r.top -= 3; r.right = r.left+9;
	{
		CClientDC dc(this);
		// bodgey way to get text size of label
		ScopeSelect<CFont> ss(&dc, font);
		CSize s = dc.GetTextExtent(_buf, buf-_buf);
		r.right += s.cx;
		r.top -= s.cy;
		label.MoveWindow(r);
	}
	label.SetWindowText(_buf);

}

//------------------------------------------------------------------------------------------
void Spectrogram::timeLimitedDraw()
{
	// don't draw if timer already triggered
	if(timer_redraw_pending) return;

	// set timer to prevent next redraw from happening too quickly
	timer_redraw_pending = true;
	SetTimer(0/*timer id*/, 75/*msec*/, NULL);

	// things to draw
	updateLabel();
	PostMessage(WM_SCROLLDRAW, 0, 0);
}


//------------------------------------------------------------------------------------------
void Spectrogram::mouseLeave()
{
	vert_line = -1;
	ReleaseCapture();
	control_active = false;

	label.MoveWindow(CRect(0,0,0,0));
	label.ShowWindow(SW_HIDE);

	timeLimitedDraw();

}

//------------------------------------------------------------------------------------------
void Spectrogram::OnMouseMove(UINT nFlags, CPoint point) 
{

	// capture mouse so we know when the pointer leaves our window
	CPoint screen_point = point;
	ClientToScreen(&screen_point);

	CRect win_rect; GetWindowRect(&win_rect);
	CRect client_rect; GetClientRect(&client_rect);

	if(WindowFromPoint(screen_point) != this) {
		mouseLeave();
	}
	else {
		SetCapture();
		if(!control_active) {
			control_active = true;

			label.BringWindowToTop();
			label.ShowWindow(SW_NORMAL);
		}
		vert_line = point.x;
		if(vert_line.isChanged()) timeLimitedDraw();
	}


	CWnd::OnMouseMove(nFlags, point);
}

//------------------------------------------------------------------------------------------
int Spectrogram::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;

	label.Create("", WS_BORDER|WS_CHILD, CRect(0,0,0,0), GetDesktopWindow());
	return 0;
}

//----------------------------------------------------------------------------------------------------
LRESULT Spectrogram::handleSetFont(WPARAM wParam, LPARAM lParam/*redraw*/)
{
	font = CFont::FromHandle((HFONT)wParam);
	label.SetFont(font);
	return Default();
}


//----------------------------------------------------------------------------------------------------
void Spectrogram::SetWindowText(LPCTSTR lpszText)
{
	name = lpszText;
}

//----------------------------------------------------------------------------------------------------
void Spectrogram::OnTimer(UINT nIDEvent) 
{
	timer_redraw_pending = false;
	updateLabel();
	PostMessage(WM_SCROLLDRAW, 0, 0);

	CWnd::OnTimer(nIDEvent);
}

//----------------------------------------------------------------------------------------------------
void Spectrogram::OnKillFocus(CWnd* pNewWnd) 
{
	mouseLeave();
	CWnd::OnKillFocus(pNewWnd);	
}
