/******************************************************************************
DtEditArray

Array of DT edits that are drawn left to right

Zero or one edits may be active - cursor is displayed in active edit

"OnKeyDown" is sent to active edit and the cursor is allowed to previous
or next edits

Particular edits may be "disabled" for displaying static text so that the edit
will never become active (i.e. gain cursor focus) 

History
	Date       Version    Programmer         Comments
	29/07/03   1.0        Darrell Tam
******************************************************************************/

#ifndef _DT_MULTI_EDIT_H_
#define _DT_MULTI_EDIT_H_

#include "DtEdit.h"
#include <vector>

class DtEditArray
{
public:
	DtEditArray();
	~DtEditArray();

	// access the number of edits, no auto redraw
	void setN(int n);
	int getN(void) const { return edit.size(); }

	bool skipEdit(int n) { return skip_edit[n]; }
	void skipEdit(int n, bool skip) { skip_edit[n] = skip; }

	// set text edit "i" to be non-edit (static) with string "s", no auto redraw
	void setStaticText(int i, const string& s);

	// set colour for all edits, no auto redraw
	void setColour(COLORREF inside, COLORREF outside);

	// set font for all edits, no auto redraw
	void setFont(CFont* font_);

	// set whether cursor is visible, no auto redraw
	void cursorVisible(bool v);
	bool cursorVisible() const { return cursor_visible; }

	// set active edit, no auto redraw
	void setActiveN(int n);

	// return active edit control number
	int getActiveN(void) const { return active_edit; }

	// move edits, skip edits that can't be changed
	// return true if wrap around end of edits
	bool prevEdit();
	bool nextEdit();

	// size in pixels of all edits drawn in a row
	CSize size(CDC *dc);

	// draw all edits onto a DC (note, size() must be called prior to these)
	void drawCursor(CDC *dc, CPoint p);
	void drawText(CDC *dc, CPoint p);

	// return true if any edit needs to be redrawn
	bool requireRedraw(void);

	// call with keypress data
	void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);

	// possible flags
	typedef enum {
		FLAG_KEY_EDIT_CHANGE,	// key caused active edit to change
		FLAG_KEY_PREV_WRAP,		// change in active edit wrapped
		FLAG_KEY_NEXT_WRAP,		// change in active edit wrapped
		N_FLAGS
	} flag_t;
	bool getFlag(flag_t f) { return flags[f]; }

	// return edit "i"
	DtEdit& get(int i) { return *(edit[limit_range(i, 0, edit.size()-1)]); }
	DtEdit& operator () (int i) { return get(i); }

	// return active edit
	DtEdit& active() { return get(active_edit); }
	DtEdit& operator () (void) { return active(); }

protected:
	// special flags
	vector<bool> flags;

	// font passed in externally
	CFont *font;

	// array of edits
	vector<auto_ptr<DtEdit> > edit;
	vector<bool> skip_edit; // whether each edit can be changed

	// which edit has keyboard focus
	int active_edit;

	// whether cursor is visible
	bool cursor_visible;

	// temporaries during draw
	vector<CSize> sz;
};


#endif
