/******************************************************************************
DtCtrl

History
	Date       Version    Programmer         Comments
	29/07/03   1.0        Darrell Tam
******************************************************************************/

#include "stdafx.h"
#include "DtCtrl.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// DtCtrl

//----------------------------------------------------------------------------------------------------
DtCtrl::DtCtrl() : font(NULL)
{
	idc = 0xffff;
	parent_draw = false;
	max_redraw_period = 75; // default to 75 msec
	parent_intercept_msgs = false;
}

//----------------------------------------------------------------------------------------------------
DtCtrl::~DtCtrl()
{
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::init()
{
	delay_draw = false;
	draw_timer_set = false;

	input_capture = false;
	lbutton_down = false;
	rbutton_down = false;
	force_redraw = false;

	desensitize_mouse = false;
	desensitize_rect = CRect(0,0,0,0);

	old_bitmap = NULL;
	mem_bitmap = NULL;
	mem_dc = NULL;
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::captureInput(bool force)
{
	//if(!force && input_capture) return;
	SetCapture();
	SetFocus();
	input_capture = true;
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::releaseInput(bool force)
{
	//if(!force && !input_capture) return;
	ReleaseCapture();
	input_capture = false;
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::redrawControl_(void)
// redraw control immediately
{
	CBitmap bm;
	CDC dc;
	prepareDraw(bm, dc);
	drawControl(); // from derived class
	endDraw();
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::redrawControl(bool _force_redraw/*=false*/)
{
	force_redraw = _force_redraw;

	// delay_draw stops draws from occurring too frequently
	if(draw_timer_set) {
		if(_force_redraw) {
			KillTimer(10000);
		}
		else {
			delay_draw = true;
			return;
		}
	}
	sendCmdMsg(NC_REDRAW);

	if(!parent_draw && (chkRedraw() || force_redraw)) redrawControl_();

	delay_draw = false;
	if(max_redraw_period) {
		draw_timer_set = true;
		SetTimer(10000, max_redraw_period, NULL);
	}
}


//----------------------------------------------------------------------------------------------------
void DtCtrl::prepareDraw(CBitmap& bitmap, CDC& dc)
// create compatible dc, bitmap and draw background stuff
{
	CClientDC cdc(this);
	mem_bitmap = &bitmap;
	mem_dc = &dc;

	mem_dc->CreateCompatibleDC(&cdc);
	mem_bitmap->CreateCompatibleBitmap(&cdc, client_rect.Width(), client_rect.Height());
	old_bitmap = mem_dc->SelectObject(mem_bitmap);

	CRect r;
	r.left = 0;
	r.right = client_rect.Width();
	r.top = 0;
	r.bottom = client_rect.Height();
	mem_dc->FillRect(r, &CBrush(0x000000UL));
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::endDraw(void) 
// copy the current memory bitmap to the screen & cleanup temp
{
	//static bool toggle = false;
	//if(toggle) mem_dc->FillRect(CRect(0,0,5,5), &CBrush(0xffffffUL));
	//toggle = !toggle;

	CClientDC(this).BitBlt(0, 0, client_rect.Width(), client_rect.Height(), mem_dc, 0, 0, SRCCOPY);
	mem_dc->SelectObject(old_bitmap);
	mem_dc->DeleteDC();

	force_redraw = false;
}


//----------------------------------------------------------------------------------------------------
BEGIN_MESSAGE_MAP(DtCtrl, CWnd)
	//{{AFX_MSG_MAP(DtCtrl)
	ON_WM_CREATE()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_PAINT()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_RBUTTONDBLCLK()
	ON_WM_SIZE()
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_SETFONT, HandleSetFont)
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// DtCtrl message handlers

//----------------------------------------------------------------------------------------------------
LRESULT DtCtrl::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
// intercept all windows messages here
{
	wp_message = message;
	wp_wParam = wParam;
	wp_lParam = lParam;
	parent_consume_msg = false;

	// parent can intercept all messages
	if(parent_intercept_msgs) sendCmdMsg(NC_INTERCEPT_MSG);

	if(parent_consume_msg) return DefWindowProc(message, wParam, lParam);
	return CWnd::WindowProc(message, wParam, lParam);
}

//----------------------------------------------------------------------------------------------------
int DtCtrl::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	init();
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;
	GetClientRect(client_rect);	
	return 0;
}

//----------------------------------------------------------------------------------------------------
bool DtCtrl::chkPoint(void)
{
	if(client_rect.PtInRect(curr_point)) {
		// mouse pointer is inside the control
		captureInput();
		return true;
	}

	// mouse has left our control
	releaseInput();
	OnMouseLeave();	// from derived class

	return false;
}


//----------------------------------------------------------------------------------------------------
void DtCtrl::OnMouseMove(UINT nFlags, CPoint point) 
{
	curr_point = point;

	if(lbutton_down) OnLButtonDrag(); // from derived class
	else if(rbutton_down) OnRButtonDrag(); // from derived class
	if(lbutton_down || rbutton_down) {
		redrawControl();
		CWnd::OnMouseMove(nFlags, point);
		return;
	}

	if(desensitize_mouse) {
		// first movement after desensitization set
		if(desensitize_rect.Width() == 0)
			desensitize_rect = CRect(point.x-4, point.y-4,point.x+4, point.y+4);

		// pointer inside desensitized rectangle, ignore movement
		if(desensitize_rect.PtInRect(point)) return;

		desensitize_rect = CRect(0,0,0,0);
		desensitize_mouse = 0;
	}
	
	if(chkPoint()) {
		OnMouseMove(); // from derived class
		CWnd::OnMouseMove(nFlags, point);
	}

	redrawControl();
}
	

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	curr_point = point;
	desensitize_mouse = false;
	if(chkPoint()) {		
		lbutton_down = true;
		OnLButtonDblClk(); // from derived class
		CWnd::OnLButtonDblClk(nFlags, point);
	}
	redrawControl();
}


//----------------------------------------------------------------------------------------------------
void DtCtrl::OnLButtonDown(UINT nFlags, CPoint point) 
{
	curr_point = point;
	desensitize_mouse = false;
	if(chkPoint()) {	
		lbutton_down = true;
		OnLButtonDown(); // from derived class
		CWnd::OnLButtonDblClk(nFlags, point);
	}
	redrawControl();
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnLButtonUp(UINT nFlags, CPoint point) 
{
	curr_point = point;
	if(!lbutton_down) return;
	lbutton_down = false;
	OnLButtonUp(); // from derived class
	chkPoint();
	redrawControl();
	sendCmdMsg(NC_BUTTONUP);
	CWnd::OnLButtonUp(nFlags, point);
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnRButtonDown(UINT nFlags, CPoint point) 
{
	curr_point = point;
	desensitize_mouse = false;
	if(chkPoint()) {
		rbutton_down = true;
		OnRButtonDown(); // from derived class
		CWnd::OnRButtonDown(nFlags, point);
	}
	redrawControl();
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	curr_point = point;
	desensitize_mouse = false;
	if(chkPoint()) {	
		rbutton_down = true;
		OnRButtonDblClk(); // from derived class
		CWnd::OnRButtonDblClk(nFlags, point);
	}
	redrawControl();
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnRButtonUp(UINT nFlags, CPoint point) 
{
	curr_point = point;
	if(!rbutton_down) return;
	rbutton_down = false;
	OnRButtonUp(); // from derived class
	chkPoint();
	redrawControl();	
	sendCmdMsg(NC_BUTTONUP);
	CWnd::OnRButtonUp(nFlags, point);
}


//----------------------------------------------------------------------------------------------------
void DtCtrl::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	redrawControl(true);
}


//----------------------------------------------------------------------------------------------------
void DtCtrl::OnSize(UINT nType, int cx, int cy) 
{
	CWnd::OnSize(nType, cx, cy);
	GetClientRect(client_rect);
}

//----------------------------------------------------------------------------------------------------
void DtCtrl::OnTimer(UINT nIDEvent) 
{
	if(nIDEvent == 10000) {
		draw_timer_set = false;
		if(delay_draw) redrawControl();
	}
	CWnd::OnTimer(nIDEvent);
}

//----------------------------------------------------------------------------------------------------
LRESULT DtCtrl::HandleSetFont(WPARAM wParam, LPARAM lParam/*redraw*/)
{
	font = CFont::FromHandle((HFONT)wParam);
	return Default();
}
